<?php
declare(strict_types=1);

function jsonResponse(array $data, int $code = 200): void {
    http_response_code($code);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

function getRequestData(): array {
    $contentType = $_SERVER['CONTENT_TYPE'] ?? '';

    if (str_contains($contentType, 'application/json')) {
        $data = json_decode(file_get_contents('php://input'), true);
        return $data ?? [];
    }

    return $_POST;
}

function getQueryParam(string $key, mixed $default = null): mixed {
    return $_GET[$key] ?? $default;
}

function validateRequired(array $data, array $fields): array {
    $errors = [];

    foreach ($fields as $field) {
        if (empty($data[$field])) {
            $errors[$field] = "El campo $field es requerido";
        }
    }

    return $errors;
}

function sanitizeString(string $value): string {
    return htmlspecialchars(strip_tags(trim($value)), ENT_QUOTES, 'UTF-8');
}

function formatDate(string $date, string $format = 'd/m/Y'): string {
    $time = strtotime($date);
    return $time ? date($format, $time) : '';
}

function formatMoney(float $amount): string {
    return '$' . number_format($amount, 0, ',', '.');
}

function getSourceLabel(string $source): string {
    return match($source) {
        'upwork' => 'Upwork',
        'workana' => 'Workana',
        'freelancer' => 'Freelancer',
        'fiverr' => 'Fiverr',
        default => ucfirst($source)
    };
}

function getStatusBadge(string $status): string {
    $classes = match($status) {
        'pending' => 'bg-yellow-500/20 text-yellow-400',
        'applied' => 'bg-blue-500/20 text-blue-400',
        'rejected' => 'bg-red-500/20 text-red-400',
        default => 'bg-gray-500/20 text-gray-400'
    };

    $labels = match($status) {
        'pending' => 'Pendiente',
        'applied' => 'Aplicado',
        'rejected' => 'Rechazado',
        default => ucfirst($status)
    };

    return "<span class=\"px-2 py-1 rounded text-xs font-medium $classes\">$labels</span>";
}

function truncateText(string $text, int $length = 100): string {
    if (mb_strlen($text) <= $length) {
        return $text;
    }
    return mb_substr($text, 0, $length) . '...';
}

function getCsrfToken(): string {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function verifyCsrfToken(string $token): bool {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}
