<?php
declare(strict_types=1);

if (!defined('ZAO_ROOT')) {
    define('ZAO_ROOT', dirname(__DIR__));
}

function getDbConnection(): ?PDO {
    static $pdo = null;

    if ($pdo !== null) {
        return $pdo;
    }

    $config = require ZAO_ROOT . '/config.php';

    if (!$config['db_enabled']) {
        return null;
    }

    $dsn = sprintf(
        'mysql:host=%s;port=%s;dbname=%s;charset=utf8mb4',
        $config['db_host'],
        $config['db_port'],
        $config['db_name']
    );

    try {
        $pdo = new PDO($dsn, $config['db_user'], $config['db_pass'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ]);
        return $pdo;
    } catch (PDOException $e) {
        error_log('DB Connection Error: ' . $e->getMessage());
        return null;
    }
}

function dbQuery(string $sql, array $params = []): ?PDOStatement {
    $pdo = getDbConnection();
    if (!$pdo) {
        return null;
    }

    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt;
    } catch (PDOException $e) {
        error_log('DB Query Error: ' . $e->getMessage() . ' | SQL: ' . $sql);
        return null;
    }
}

function dbFetchAll(string $sql, array $params = []): array {
    $stmt = dbQuery($sql, $params);
    return $stmt ? $stmt->fetchAll() : [];
}

function dbFetchOne(string $sql, array $params = []): ?array {
    $stmt = dbQuery($sql, $params);
    return $stmt ? $stmt->fetch() : null;
}

function dbInsert(string $table, array $data): ?int {
    $pdo = getDbConnection();
    if (!$pdo) {
        return null;
    }

    $columns = implode(', ', array_keys($data));
    $placeholders = implode(', ', array_fill(0, count($data), '?'));

    $sql = "INSERT INTO $table ($columns) VALUES ($placeholders)";

    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute(array_values($data));
        return (int) $pdo->lastInsertId();
    } catch (PDOException $e) {
        error_log('DB Insert Error: ' . $e->getMessage());
        return null;
    }
}

function dbUpdate(string $table, array $data, array $where): ?int {
    $pdo = getDbConnection();
    if (!$pdo) {
        return null;
    }

    $set = implode(', ', array_map(fn($k) => "$k = ?", array_keys($data)));
    $whereClause = implode(' AND ', array_map(fn($k) => "$k = ?", array_keys($where)));

    $sql = "UPDATE $table SET $set WHERE $whereClause";

    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute([...array_values($data), ...array_values($where)]);
        return $stmt->rowCount();
    } catch (PDOException $e) {
        error_log('DB Update Error: ' . $e->getMessage());
        return null;
    }
}

function dbDelete(string $table, array $where): ?int {
    $pdo = getDbConnection();
    if (!$pdo) {
        return null;
    }

    $whereClause = implode(' AND ', array_map(fn($k) => "$k = ?", array_keys($where)));
    $sql = "DELETE FROM $table WHERE $whereClause";

    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute(array_values($where));
        return $stmt->rowCount();
    } catch (PDOException $e) {
        error_log('DB Delete Error: ' . $e->getMessage());
        return null;
    }
}
