<?php
declare(strict_types=1);

if (session_status() === PHP_SESSION_NONE) {
    $config = require ZAO_ROOT . '/config.php';
    session_name($config['session_name']);
    session_start();
}

function isLoggedIn(): bool {
    return isset($_SESSION['user_id']) && $_SESSION['user_id'] > 0;
}

function requireAuth(): void {
    if (!isLoggedIn()) {
        header('Location: /index.php');
        exit;
    }
}

function getCurrentUserId(): ?int {
    return $_SESSION['user_id'] ?? null;
}

function getCurrentUser(): ?array {
    if (!isLoggedIn()) {
        return null;
    }

    static $user = null;
    if ($user !== null) {
        return $user;
    }

    $user = dbFetchOne(
        'SELECT id, email, name, created_at FROM users WHERE id = ?',
        [getCurrentUserId()]
    );

    return $user;
}

function login(string $email, string $password, bool $remember = false): bool {
    $user = dbFetchOne('SELECT * FROM users WHERE email = ?', [$email]);

    if (!$user) {
        return false;
    }

    if (password_verify($password, $user['password'])) {
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['user_email'] = $user['email'];
        $_SESSION['user_name'] = $user['name'];

        if ($remember) {
            $token = bin2hex(random_bytes(32));
            $expires = time() + (30 * 86400);

            setcookie('remember_token', $token, [
                'expires' => $expires,
                'path' => '/',
                'httponly' => true,
                'samesite' => 'strict'
            ]);
        }

        return true;
    }

    return false;
}

function logout(): void {
    $_SESSION = [];

    if (isset($_COOKIE[session_name()])) {
        setcookie(session_name(), '', time() - 42000, '/');
    }

    if (isset($_COOKIE['remember_token'])) {
        setcookie('remember_token', '', time() - 42000, '/');
    }

    session_destroy();
}

function hashPassword(string $password): string {
    return password_hash($password, PASSWORD_DEFAULT);
}
