<?php
declare(strict_types=1);
define('ZAO_ROOT', dirname(__DIR__));

require_once ZAO_ROOT . '/includes/db.php';
require_once ZAO_ROOT . '/includes/auth.php';
require_once ZAO_ROOT . '/includes/functions.php';

header('Content-Type: application/json');

requireAuth();

$userId = getCurrentUserId();
$method = $_SERVER['REQUEST_METHOD'];
$id = (int)($_GET['id'] ?? 0);

if ($method === 'GET') {
    if ($id) {
        $proposal = dbFetchOne(
            "SELECT * FROM proposals WHERE id = ? AND user_id = ?",
            [$id, $userId]
        );

        if (!$proposal) {
            jsonResponse(['error' => 'Not found'], 404);
        }

        jsonResponse($proposal);
    }

    $where = ['user_id = ?'];
    $params = [$userId];

    foreach (['source', 'status'] as $field) {
        if (!empty($_GET[$field])) {
            $where[] = "$field = ?";
            $params[] = $_GET[$field];
        }
    }

    if (!empty($_GET['search'])) {
        $where[] = '(title LIKE ? OR description LIKE ?)';
        $params[] = "%{$_GET['search']}%";
        $params[] = "%{$_GET['search']}%";
    }

    if (!empty($_GET['date_from'])) {
        $where[] = 'DATE(created_at) >= ?';
        $params[] = $_GET['date_from'];
    }

    if (!empty($_GET['date_to'])) {
        $where[] = 'DATE(created_at) <= ?';
        $params[] = $_GET['date_to'];
    }

    $whereClause = implode(' AND ', $where);
    $page = max(1, (int)($_GET['page'] ?? 1));
    $perPage = (int)($_GET['per_page'] ?? 20);
    $offset = ($page - 1) * $perPage;

    $proposals = dbFetchAll(
        "SELECT * FROM proposals WHERE $whereClause ORDER BY created_at DESC LIMIT $perPage OFFSET $offset",
        $params
    );

    $total = dbFetchOne("SELECT COUNT(*) as count FROM proposals WHERE $whereClause", $params)['count'] ?? 0;

    jsonResponse([
        'data' => $proposals,
        'pagination' => [
            'page' => $page,
            'per_page' => $perPage,
            'total' => $total,
            'pages' => max(1, (int)ceil($total / $perPage))
        ]
    ]);
}

if ($method === 'POST') {
    $data = getRequestData();

    $errors = validateRequired($data, ['title', 'source']);
    if (!empty($errors)) {
        jsonResponse(['errors' => $errors], 400);
    }

    $proposalId = dbInsert('proposals', [
        'title' => $data['title'],
        'description' => $data['description'] ?? '',
        'source' => $data['source'],
        'url' => $data['url'] ?? '',
        'budget' => $data['budget'] ?? null,
        'status' => $data['status'] ?? 'pending',
        'user_id' => $userId,
    ]);

    if ($proposalId) {
        $proposal = dbFetchOne("SELECT * FROM proposals WHERE id = ?", [$proposalId]);
        jsonResponse($proposal, 201);
    }

    jsonResponse(['error' => 'Error creating proposal'], 500);
}

if ($method === 'PUT') {
    if (!$id) {
        jsonResponse(['error' => 'ID required'], 400);
    }

    $data = getRequestData();

    $existing = dbFetchOne("SELECT id FROM proposals WHERE id = ? AND user_id = ?", [$id, $userId]);
    if (!$existing) {
        jsonResponse(['error' => 'Not found'], 404);
    }

    $updateData = [];

    foreach (['title', 'description', 'source', 'url', 'budget', 'status'] as $field) {
        if (array_key_exists($field, $data)) {
            $updateData[$field] = $data[$field];
        }
    }

    if (!empty($updateData)) {
        dbUpdate('proposals', $updateData, ['id' => $id, 'user_id' => $userId]);
    }

    $proposal = dbFetchOne("SELECT * FROM proposals WHERE id = ?", [$id]);
    jsonResponse($proposal);
}

if ($method === 'DELETE') {
    if (!$id) {
        jsonResponse(['error' => 'ID required'], 400);
    }

    $existing = dbFetchOne("SELECT id FROM proposals WHERE id = ? AND user_id = ?", [$id, $userId]);
    if (!$existing) {
        jsonResponse(['error' => 'Not found'], 404);
    }

    dbDelete('proposals', ['id' => $id, 'user_id' => $userId]);
    jsonResponse(['success' => true]);
}

jsonResponse(['error' => 'Method not allowed'], 405);
