# Plan: ZAO - Sistema de Gestión de Propuestas

## Contexto
Aplicación web para recibir y gestionar propuestas de desarrollo web filtradas de plataformas como Upwork, Workana, etc. Conectada a clawdbot/bot de telegram.

## Tech Stack (Plain/Vanilla)
- **Backend**: PHP 8+ (vanilla, sin framework)
- **Frontend**: HTML5
- **Estilos**: CSS3 + Tailwind CSS (via CDN)
- **JavaScript**: Vanilla ES6+
- **Base de datos**: MySQL/MariaDB
- **Autenticación**: Sesiones nativas de PHP

## Especificaciones de Diseño
- Dark mode exclusivo
- Paleta: grises oscuros, negros, acentos en `#0e448f`
- Sin box-shadows, sin grains
- Diseño moderno y minimalista inspirado en shadcn/ui

## Estructura de Archivos

```
/zao
├── index.php              → Login
├── dashboard.php          → Dashboard principal
├── proposals.php          → Lista de propuestas
├── templates.php          → Textos repetitivos
├── filters.php            → Configuración de filtros
├── logout.php             → Cerrar sesión
├── /api
│   ├── auth.php           → Login/Logout endpoints
│   ├── proposals.php      → CRUD propuestas
│   ├── templates.php      → CRUD templates
│   ├── filters.php        → CRUD filtros
│   └── webhook.php        → Recepción desde clawdbot
├── /includes
│   ├── db.php             → Conexión DB
│   ├── auth.php           → Middleware de auth
│   ├── functions.php      → Utilidades
│   └── header.php         → Header común
├── /assets
│   ├── css/
│   │   └── styles.css     → Estilos custom
│   └── js/
│       ├── app.js         → JS global
│       ├── dashboard.js
│       ├── proposals.js
│       └── templates.js
└── /sql
    └── schema.sql         → Estructura DB
```

## Esquema de Base de Datos

```sql
CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(255) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    name VARCHAR(100),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE proposals (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(500) NOT NULL,
    description TEXT,
    source VARCHAR(50) NOT NULL,  -- upwork, workana, etc
    url VARCHAR(500),
    budget DECIMAL(10,2),
    status VARCHAR(20) DEFAULT 'pending', -- pending, applied, rejected
    user_id INT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id)
);

CREATE TABLE templates (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    content TEXT NOT NULL,
    category VARCHAR(100),
    user_id INT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id)
);

CREATE TABLE filters (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    criteria JSON NOT NULL,
    is_active BOOLEAN DEFAULT TRUE,
    user_id INT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id)
);
```

## Funcionalidades por Sección

### 1. Login (`index.php`)
- Form email + contraseña
- Checkbox "recordarme"
- Validación JS + PHP
- Redirección a dashboard

### 2. Dashboard (`dashboard.php`)
- Sidebar de navegación
- Cards con métricas:
  - Propuestas recibidas hoy
  - Propuestas esta semana
  - Por fuente (Upwork, Workana)
  - Pendientes vs aplicadas
- Gráfico simple (Chart.js o similar)
- Tabla de propuestas recientes

### 3. Propuestas (`proposals.php`)
- Tabla con paginación
- Filtros:
  - Rango fechas (date picker)
  - Fuente (select)
  - Estado (select)
  - Buscador título
- Acciones: ver, editar estado, eliminar
- Modal para ver detalle

### 4. Templates (`templates.php`)
- Lista con búsqueda
- Crear/editar/eliminar
- Modal para edición
- Botón copiar al clipboard

### 5. Filtros (`filters.php`)
- Lista de filtros
- Crear/editar con:
  - Palabras clave include/exclude
  - Rango presupuesto
  - Skills
- Toggle activo/inactivo

### 6. Logout (`logout.php`)
- Destruir sesión
- Redirigir a login

## Componentes UI Reutilizables (CSS)
- Botones (primario, secundario, danger)
- Inputs (text, email, password, select)
- Cards
- Modals
- Tables
- Badges
- Sidebar

## API Endpoints

### Auth
- POST `/api/auth.php` → Login
- POST `/api/auth.php?action=logout` → Logout

### Propuestas
- GET `/api/proposals.php` → Listar (con filtros por query params)
- POST `/api/proposals.php` → Crear
- PUT `/api/proposals.php` → Actualizar
- DELETE `/api/proposals.php` → Eliminar

### Templates
- GET `/api/templates.php` → Listar
- POST `/api/templates.php` → Crear
- PUT `/api/templates.php` → Actualizar
- DELETE `/api/templates.php` → Eliminar

### Filtros
- GET `/api/filters.php` → Listar
- POST `/api/filters.php` → Crear
- PUT `/api/filters.php` → Actualizar
- DELETE `/api/filters.php` → Eliminar

### Webhook
- POST `/api/webhook.php` → Recibir propuestas desde clawdbot

## Plan de Implementación

### Fase 1: Estructura Base
1. Crear estructura de directorios
2. Configurar base de datos MySQL
3. Crear archivo de conexión DB
4. Crear sistema de sesiones y auth
5. Estilos base con Tailwind + CSS custom

### Fase 2: Login
1. Página login (index.php)
2. API auth
3. Middleware de protección

### Fase 3: Layout Principal
1. Sidebar de navegación
2. Header común
3. Estructura base de páginas interiores

### Fase 4: Dashboard
1. Página dashboard.php
2. API de estadísticas
3. Gráficos simples

### Fase 5: Propuestas
1. Página proposals.php
2. Tabla con paginación JS
3. Filtros (fecha, source, estado)
4. CRUD completo

### Fase 6: Templates
1. Página templates.php
2. CRUD + modal
3. Copiar al clipboard

### Fase 7: Filtros
1. Página filters.php
2. CRUD de configuración
3. Interface flexible

## Consideraciones clawdbot/telegram
- Webhook endpoint `/api/webhook.php`
- JSON response estandarizado
- Token de autenticación para webhook
